import subprocess
import requests
import threading
import logging
import platform
# time 模块不再需要导入，因为它已在 serial_listener 中使用

logger = logging.getLogger(__name__)

class ActionExecutor:
    """
    根据配置中定义的类型执行相应的操作。
    此模块不再处理去重逻辑，只关注动作的执行。
    """
    def __init__(self):
        # 移除去重相关的属性
        pass 

    def execute_action(self, action_config):
        """
        执行单个动作配置。
        """
        action_type = action_config.get('type')
        command = action_config.get('command')
        message = action_config.get('message')
        data = action_config.get('data', 'N/A') # 用于日志显示

        # 在新线程中执行动作，避免阻塞主监听线程
        thread = threading.Thread(target=self._run_action_in_thread, 
                                  args=(action_type, command, message, data))
        thread.daemon = True # 设置为守护线程
        thread.start()

    def _run_action_in_thread(self, action_type, command, message, data):
        """
        在新线程中执行的具体动作逻辑。
        """
        try:
            if action_type == 'curl':
                logger.info(f"执行 Curl 操作 (数据: {data}): {command}")
                self._execute_curl(command)
            elif action_type == 'execute':
                logger.info(f"执行外部命令 (数据: {data}): {command}")
                self._execute_command(command)
            elif action_type == 'log':
                logger.info(f"仅记录日志 (数据: {data}): {message}")
            else:
                logger.warning(f"不支持的动作类型: {action_type} (数据: {data})")
        except Exception as e:
            logger.error(f"执行动作 '{action_type}' (数据: {data}) 时发生错误: {e}")

    def _execute_curl(self, url):
        """
        发送 HTTP GET 请求。
        """
        try:
            response = requests.get(url, timeout=5) # 5秒超时
            response.raise_for_status() # 对 4xx/5xx 状态码抛出异常
            logger.info(f"Curl 请求成功: {url}, 状态码: {response.status_code}")
        except requests.exceptions.RequestException as e:
            logger.error(f"Curl 请求失败: {url}, 错误: {e}")

    def _execute_command(self, cmd):
        """
        执行外部系统命令。
        """
        try:
            if platform.system() == "Windows":
                # DETACHED_PROCESS ensures the process runs in its own console, not attached to current.
                # close_fds=True is for consistency across platforms and handles file descriptors.
                subprocess.Popen(cmd, shell=True, creationflags=subprocess.DETACHED_PROCESS, close_fds=True)
                logger.info(f"已尝试启动 Windows 命令/程序: {cmd}")
            else:
                # preexec_fn=os.setsid detaches the child process from its parent, creating a new session.
                # This makes it independent of the parent's lifetime.
                subprocess.Popen(cmd, shell=True, preexec_fn=os.setsid)
                logger.info(f"已尝试启动 Unix/Linux/macOS 命令/程序: {cmd}")
        except FileNotFoundError:
            logger.error(f"命令 '{cmd}' 未找到。请检查命令路径和环境变量。")
        except Exception as e:
            logger.error(f"执行命令 '{cmd}' 时发生错误: {e}")