import json
import logging
import os

# 设置日志
logger = logging.getLogger(__name__)

class ConfigManager:
    """
    负责加载和管理应用程序的配置文件。
    """
    def __init__(self, config_file='config.json'):
        self.config_file = config_file
        self.config = self._load_config()

    def _load_config(self):
        """加载配置文件"""
        if not os.path.exists(self.config_file):
            logger.error(f"配置文件 '{self.config_file}' 不存在。请创建它。")
            raise FileNotFoundError(f"Config file not found: {self.config_file}")

        try:
            with open(self.config_file, 'r', encoding='utf-8') as f:
                config_data = json.load(f)
            logger.info(f"配置文件 '{self.config_file}' 加载成功。")
            return self._validate_config(config_data)
        except json.JSONDecodeError as e:
            logger.error(f"配置文件 '{self.config_file}' JSON 解析错误: {e}")
            raise ValueError(f"Invalid JSON in config file: {e}")
        except Exception as e:
            logger.error(f"加载配置文件时发生未知错误: {e}")
            raise

    def _validate_config(self, config_data):
        """验证配置文件的基本结构和必填项。"""
        # 验证 serial_settings
        serial_settings = config_data.get('serial_settings')
        if not isinstance(serial_settings, dict):
            raise ValueError("配置文件中缺少 'serial_settings' 或格式不正确。")
        if not isinstance(serial_settings.get('baud_rate'), int):
            raise ValueError("'serial_settings' 中缺少 'baud_rate' 或类型不正确。")
        if 'timeout' not in serial_settings:
            serial_settings['timeout'] = 1 # 默认值

        # 验证 actions
        actions = config_data.get('actions')
        if not isinstance(actions, list):
            raise ValueError("配置文件中缺少 'actions' 或格式不正确。")
        for action in actions:
            if not isinstance(action, dict) or 'data' not in action or 'type' not in action:
                raise ValueError("每个 'action' 必须包含 'data' 和 'type' 字段。")
            if action['type'] not in ['curl', 'execute', 'log']:
                raise ValueError(f"不支持的 'action' 类型: {action['type']}")
            if action['type'] == 'curl' and 'command' not in action:
                raise ValueError("类型为 'curl' 的 'action' 必须包含 'command' 字段。")
            if action['type'] == 'execute' and 'command' not in action:
                raise ValueError("类型为 'execute' 的 'action' 必须包含 'command' 字段。")
            if action['type'] == 'log' and 'message' not in action:
                raise ValueError("类型为 'log' 的 'action' 必须包含 'message' 字段。")

        # 验证 log_settings
        log_settings = config_data.get('log_settings')
        if not isinstance(log_settings, dict):
            raise ValueError("配置文件中缺少 'log_settings' 或格式不正确。")
        if not isinstance(log_settings.get('enable_file_logging'), bool):
            raise ValueError("'log_settings' 中缺少 'enable_file_logging' 或类型不正确。")
        if log_settings.get('enable_file_logging') and 'log_file_path' not in log_settings:
             raise ValueError("'log_settings' 中 'enable_file_logging' 为 True 时，必须提供 'log_file_path'。")

        return config_data

    def get_serial_settings(self):
        """获取串口配置"""
        return self.config.get('serial_settings', {})

    def get_actions(self):
        """获取动作列表"""
        return self.config.get('actions', [])

    def get_log_settings(self):
        """获取日志配置"""
        return self.config.get('log_settings', {})