import logging
import logging.handlers # 确保导入了 logging.handlers 模块
import sys
import os
import time

from config_manager import ConfigManager
from serial_listener import SerialListener
from action_executor import ActionExecutor

# --- 配置日志 ---
def setup_logging(log_settings):
    """
    配置程序的日志记录。
    """
    # 根日志器
    root_logger = logging.getLogger()
    root_logger.setLevel(logging.INFO) # 默认级别为 INFO，调试时可改为 DEBUG
                                       # 如果要看去重时的DEBUG信息，可以改为 logging.DEBUG

    # 清除旧的处理器，避免重复
    for handler in root_logger.handlers[:]:
        root_logger.removeHandler(handler)

    # 定义新的时间格式和日志格式
    log_format = '%(asctime)s - %(levelname)s - %(name)s - %(message)s'
    date_format = '%m-%d %H:%M:%S' # 月-日 时:分:秒

    # 1. 控制台输出处理器
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setFormatter(logging.Formatter(log_format, datefmt=date_format))
    root_logger.addHandler(console_handler)

    # 2. 文件输出处理器（如果启用）
    if log_settings.get('enable_file_logging', False):
        log_file_path = log_settings.get('log_file_path', '433_receiver.log')
        
        # 估算10000行日志的文件大小
        # 假设每行日志平均100-150字节，10000行就是 1MB 到 1.5MB。
        # 这里设置为 2MB，留有余量。
        MAX_LOG_FILE_SIZE_BYTES = 2 * 1024 * 1024  # 2 MB
        BACKUP_LOG_FILES = 5 # 保留 5 个旧的日志文件 (例如：433_receiver.log.1, .2, ...)

        try:
            # 使用 RotatingFileHandler 实现日志文件自动轮转
            # maxBytes: 单个日志文件达到此大小时进行轮转
            # backupCount: 保留的旧日志文件数量
            # encoding: 指定编码以支持中文字符
            file_handler = logging.handlers.RotatingFileHandler(
                log_file_path, 
                maxBytes=MAX_LOG_FILE_SIZE_BYTES, 
                backupCount=BACKUP_LOG_FILES, 
                encoding='utf-8'
            )
            file_handler.setFormatter(logging.Formatter(log_format, datefmt=date_format))
            root_logger.addHandler(file_handler)
            root_logger.info(f"日志将同时写入文件: {os.path.abspath(log_file_path)}")
            root_logger.info(f"日志文件将自动轮转，最大 {MAX_LOG_FILE_SIZE_BYTES / (1024*1024):.1f}MB，保留 {BACKUP_LOG_FILES} 个备份。")
        except Exception as e:
            root_logger.error(f"无法设置文件日志到 '{log_file_path}': {e}. 将仅输出到控制台。")


def main():
    """
    程序主入口点。
    """
    config = None
    try:
        config_manager = ConfigManager()
        config = config_manager.config
        
        setup_logging(config_manager.get_log_settings())

    except Exception as e:
        logging.error(f"程序启动失败：配置文件加载或验证错误 - {e}")
        sys.exit(1)

    logger = logging.getLogger(__name__)

    serial_settings = config_manager.get_serial_settings()
    actions_config = config_manager.get_actions()

    action_executor = ActionExecutor()

    def data_received_callback(hex_data):
        logger.info("-" * 50)
        logger.info(f"接收到新数据 (Hex): {hex_data}")
        found_action = False
        
        for action in actions_config:
            expected_data = action['data'].lower()
            
            if len(hex_data) >= len(expected_data) and hex_data.startswith(expected_data):
                logger.info(f"匹配到动作 (前缀匹配): {expected_data}")
                action_executor.execute_action(action)
                found_action = True
                break
            elif hex_data.lower() == expected_data:
                logger.info(f"匹配到动作 (精确匹配): {expected_data}")
                action_executor.execute_action(action)
                found_action = True
                break
            
        if not found_action:
            logger.info(f"未找到与数据 '{hex_data}' 匹配的动作。")

    debounce_interval = serial_settings.get('debounce_interval_seconds', 1.5) 
    serial_listener = SerialListener(serial_settings, data_received_callback, debounce_interval_seconds=debounce_interval)

    try:
        serial_listener.start_listening()
        logger.info("433MHz 串口监听程序已启动。按 Ctrl+C 退出。")
        
        while True:
            time.sleep(1) 
    except KeyboardInterrupt:
        logger.info("\n收到退出信号 (Ctrl+C)。正在关闭程序...")
    except Exception as e:
        logger.critical(f"程序运行过程中发生严重错误: {e}")
    finally:
        serial_listener.stop_listening()
        logger.info("程序已安全退出。")

if __name__ == "__main__":
    main()