import serial
import serial.tools.list_ports
import time
import threading
import logging

logger = logging.getLogger(__name__)

class SerialListener:
    """
    负责自动检测串口，连接并监听来自 433MHz 模块的数据。
    在接收到数据后，进行初步的去重处理，再将有效数据传递给回调函数。
    """
    def __init__(self, serial_settings, data_callback, debounce_interval_seconds=1.0):
        self.serial_settings = serial_settings
        self.data_callback = data_callback # 接收到去重后数据后的回调函数
        self.debounce_interval = debounce_interval_seconds # 去重时间间隔（秒）
        self.last_handled_time = {} # 存储每个数据包上次被处理的时间戳

        self.ser = None
        self._running = threading.Event() # 用于控制线程的停止

    def _find_serial_port(self):
        """
        根据配置查找合适的串口。
        """
        port_filter = self.serial_settings.get('port_description_filter')
        
        logger.info("正在搜索可用串口...")
        available_ports = serial.tools.list_ports.comports()
        
        if not available_ports:
            logger.warning("未找到任何串口设备。")
            return None

        logger.info("可用串口列表:")
        for p in available_ports:
            logger.info(f"  端口: {p.device}, 描述: {p.description}, 硬件ID: {p.hwid}")
            # 优先匹配过滤器
            if port_filter and port_filter in p.description:
                logger.info(f"找到匹配的串口: {p.device} (根据过滤器 '{port_filter}')")
                return p.device
        
        # 如果有过滤器但没找到匹配的，或者没有过滤器，尝试使用第一个可用串口
        if port_filter: # 如果指定了过滤器但没找到匹配项
            logger.warning(f"未找到描述包含 '{port_filter}' 的串口。尝试使用第一个可用串口。")

        first_port = available_ports[0].device
        logger.info(f"使用第一个可用串口: {first_port}")
        return first_port

    def start_listening(self):
        """
        启动串口监听线程。
        """
        if self._running.is_set():
            logger.warning("串口监听器已在运行中。")
            return

        self._running.set()
        
        # 在新线程中运行监听，避免阻塞主程序
        self.listener_thread = threading.Thread(target=self._listen_loop)
        self.listener_thread.daemon = True # 设置为守护线程，主程序退出时它也会退出
        self.listener_thread.start()
        logger.info("串口监听线程已启动。")

    def stop_listening(self):
        """
        停止串口监听线程。
        """
        if self._running.is_set():
            self._running.clear()
            if self.listener_thread and self.listener_thread.is_alive():
                self.listener_thread.join(timeout=2) # 等待线程结束
                if self.listener_thread.is_alive():
                    logger.warning("串口监听线程未能正常停止。")
            if self.ser and self.ser.is_open:
                self.ser.close()
                logger.info("串口已关闭。")
            logger.info("串口监听已停止。")
        else:
            logger.warning("串口监听器未在运行。")

    def _listen_loop(self):
        """
        串口监听主循环。包含去重逻辑。
        """
        port = self._find_serial_port()
        if not port:
            self._running.clear()
            return

        baud_rate = self.serial_settings.get('baud_rate')
        timeout = self.serial_settings.get('timeout', 1)

        while self._running.is_set():
            # 检查串口是否已连接或需要重新连接
            if not self.ser or not self.ser.is_open:
                try:
                    self.ser = serial.Serial(port, baud_rate, timeout=timeout)
                    logger.info(f"成功连接到串口: {port} (波特率: {baud_rate})")
                except serial.SerialException as e:
                    logger.error(f"无法连接到串口 {port}: {e}. 2秒后重试...")
                    time.sleep(2)
                    continue
                except Exception as e:
                    logger.error(f"打开串口时发生未知错误: {e}. 2秒后重试...")
                    time.sleep(2)
                    continue

            try:
                if self.ser.in_waiting > 0:
                    data = self.ser.read(self.ser.in_waiting)
                    hex_data = data.hex() # 将字节数据转换为十六进制字符串
                    
                    current_time = time.time()
                    
                    # --- 去重逻辑 ---
                    # 只有当数据是新的，或者距离上次处理已经超过去重间隔时，才进行处理
                    if hex_data not in self.last_handled_time or \
                       (current_time - self.last_handled_time[hex_data]) >= self.debounce_interval:
                        
                        self.last_handled_time[hex_data] = current_time # 更新时间戳
                        # 这里不再直接打印 "接收到新数据"，而是通过回调让 main.py 统一处理，并在前面加横线
                        self.data_callback(hex_data) # 调用回调函数处理有效数据
                    else:
                        # 这是一个被去重的重复信号，仅在 DEBUG 级别打印
                        logger.debug(f"数据 '{hex_data}' 在去重间隔 ({self.debounce_interval}s) 内，已忽略。")

            except serial.SerialException as e:
                logger.error(f"串口读取错误或断开连接: {e}. 尝试重新连接...")
                if self.ser and self.ser.is_open:
                    self.ser.close()
                self.ser = None # 重置串口对象，以便重新连接
                time.sleep(2) # 等待一段时间再重试
            except Exception as e:
                logger.error(f"读取串口数据时发生未知错误: {e}")
                time.sleep(1) # 短暂等待，避免无限循环错误

            time.sleep(0.01) # 短暂延时，避免CPU占用过高